unit CustomTrackBar;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls;

const SAVE_MINIMUM = -1000000000;
      SAVE_MAXIMUM =  1000000000;

type

  TTrackBarPointerPanel = class(TPanel)
    protected
      procedure MouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    public
      Catched:boolean;

      MousePos:TPoint;

      constructor Create(AOwner:TComponent);override;
  end;

  TCustomTrackBar = class(TPanel)
    private
      FMinValue:integer;
      FMaxValue:integer;
      FValue:integer;
      FStep:integer;

      FOnChange:TNotifyEvent;
      FOnResize:TNotifyEvent;

      LeftArrowPanel:TPanel;
      RightArrowPanel:TPanel;
      TrackPointerPanel:TTrackBarPointerPanel;

      procedure PanelOnResize(Sender:TObject);

      procedure LeftArrowPanelOnMouseDown(Sender:TObject;
                                          Button:TMouseButton;
                                          Shift:TShiftState;
                                          X,Y:Integer);
      procedure RightArrowPanelOnMouseDown(Sender:TObject;
                                           Button:TMouseButton;
                                           Shift:TShiftState;
                                           X,Y:Integer);
      procedure TrackPointerPanelOnMouseDown(Sender:TObject;
                                             Button:TMouseButton;
                                             Shift:TShiftState;
                                             X,Y:Integer);
      procedure TrackPointerPanelOnMouseMove(Sender:TObject;
                                             Shift:TShiftState;
                                             X,Y:Integer);
      procedure TrackPointerPanelOnMouseUp(Sender:TObject;
                                           Button:TMouseButton;
                                           Shift:TShiftState;
                                           X,Y:Integer);

      function GetCaption:TCaption;
      procedure SetCaption(NewCaption:TCaption);
      function GetFont:TFont;
      procedure SetFont(NewFont:TFont);

      function GetLeftArrowColor:TColor;
      procedure SetLeftArrowColor(NewColor:TColor);
      function GetRightArrowColor:TColor;
      procedure SetRightArrowColor(NewColor:TColor);
      function GetTrackPointerColor:TColor;
      procedure SetTrackPointerColor(NewColor:TColor);

      function GetLeftArrowWidth:integer;
      procedure SetLeftArrowWidth(NewWidth:integer);
      function GetRightArrowWidth:integer;
      procedure SetRightArrowWidth(NewWidth:integer);
      function GetTrackPointerWidth:integer;
      procedure SetTrackPointerWidth(NewWidth:integer);

      procedure SetMinValue(NewValue:integer);
      procedure SetMaxValue(NewValue:integer);
      procedure SetValue(NewValue:integer);
    protected

    public

      constructor Create(AOwner:TComponent);override;
      destructor Destroy;override;

      procedure DoOnResize;
    published
      property Caption:TCaption read GetCaption write SetCaption;
      property Font:TFont read GetFont write SetFont;
      property LeftArrowColor:TColor read GetLeftArrowColor write SetLeftArrowColor;
      property RightArrowColor:TColor read GetRightArrowColor write SetRightArrowColor;
      property TrackPointerColor:TColor read GetTrackPointerColor write SetTrackPointerColor;
      property LeftArrowWidth:integer read GetLeftArrowWidth write SetLeftArrowWidth;
      property RightArrowWidth:integer read GetRightArrowWidth write SetRightArrowWidth;
      property TrackPointerWidth:integer read GetTrackPointerWidth write SetTrackPointerWidth;
      property MinValue:integer read FMinValue write SetMinValue;
      property MaxValue:integer read FMaxValue write SetMaxValue;
      property Value:integer read FValue write SetValue;
      property Step:integer read FStep write FStep;
      property OnChange:TNotifyEvent read FOnChange write FOnChange;
      property OnResize:TNotifyEvent read FOnResize write FOnResize;

  end;

procedure Register;

implementation

//********  TTrackBarPointerPanel  *********************************************

constructor TTrackBarPointerPanel.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);

  Catched:=False;
end;

procedure TTrackBarPointerPanel.MouseLeave(var Message: TMessage);
begin
  Catched:=False;
end;




//********  TCustomTrackBar  *********************************************

constructor TCustomTrackBar.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  inherited Caption:='';
  Color:=clBlack;
  Height:=15;

  LeftArrowPanel:=TPanel.Create(Self);
  LeftArrowPanel.Parent:=Self;
  LeftArrowPanel.Width:=10;
  LeftArrowPanel.Caption:='<';
  LeftArrowPanel.Font.Color:=clWhite;
  LeftArrowPanel.Color:=clBlack;

  RightArrowPanel:=TPanel.Create(Self);
  RightArrowPanel.Parent:=Self;
  RightArrowPanel.Width:=10;
  RightArrowPanel.Caption:='>';
  RightArrowPanel.Font.Color:=clWhite;
  RightArrowPanel.Color:=clBlack;

  TrackPointerPanel:=TTrackBarPointerPanel.Create(Self);
  TrackPointerPanel.Parent:=Self;
  TrackPointerPanel.Width:=10;
  TrackPointerPanel.Caption:='|';
  TrackPointerPanel.Font.Color:=clWhite;
  TrackPointerPanel.Color:=clBlack;

  LeftArrowPanel.OnMouseDown:=LeftArrowPanelOnMouseDown;
  RightArrowPanel.OnMouseDown:=RightArrowPanelOnMouseDown;
  TrackPointerPanel.OnMouseDown:=TrackPointerPanelOnMouseDown;
  TrackPointerPanel.OnMouseMove:=TrackPointerPanelOnMouseMove;
  TrackPointerPanel.OnMouseUp:=TrackPointerPanelOnMouseUp;



  FMinValue:=SAVE_MINIMUM;
  FMaxValue:=SAVE_MAXIMUM;
  FValue:=0;
  FStep:=1;

  DoOnResize;

  inherited OnResize:=PanelOnResize;

end;

destructor TCustomTrackBar.Destroy;
begin
  LeftArrowPanel.Free;
  RightArrowPanel.Free;
  TrackPointerPanel.Free;

  inherited Destroy;
end;

procedure TCustomTrackBar.PanelOnResize(Sender:TObject);
begin
  DoOnResize;

  if Assigned(OnResize) then OnResize(Self);
end;

procedure TCustomTrackBar.LeftArrowPanelOnMouseDown(Sender:TObject;
                                                    Button:TMouseButton;
                                                    Shift:TShiftState;
                                                    X,Y:Integer);
begin
  SetValue(Value-Step);
end;

procedure TCustomTrackBar.RightArrowPanelOnMouseDown(Sender:TObject;
                                                     Button:TMouseButton;
                                                     Shift:TShiftState;
                                                     X,Y:Integer);
begin
  SetValue(Value+Step);
end;

procedure TCustomTrackBar.TrackPointerPanelOnMouseDown(Sender:TObject;
                                                       Button:TMouseButton;
                                                       Shift:TShiftState;
                                                       X,Y:Integer);
begin
  TrackPointerPanel.Catched:=True;

  TrackPointerPanel.MousePos.x:=X;
end;

procedure TCustomTrackBar.TrackPointerPanelOnMouseMove(Sender:TObject;
                                                       Shift:TShiftState;
                                                       X,Y:Integer);
begin
  if (TrackPointerPanel.Left+(X-TrackPointerPanel.MousePos.x) >
      Width-(RightArrowPanel.Width+TrackPointerPanel.Width))    or
     (TrackPointerPanel.Left+(X-TrackPointerPanel.MousePos.x) <
      LeftArrowPanel.Width)                                     or
     not TrackPointerPanel.Catched                              then Exit;

  SetValue(Round(((MaxValue-MinValue)/
                  (Width-(LeftArrowPanel.Width+RightArrowPanel.Width+TrackPointerPanel.Width)))*
                 ((TrackPointerPanel.Left+(X-TrackPointerPanel.MousePos.x))-LeftArrowPanel.Width))+MinValue);
end;

procedure TCustomTrackBar.TrackPointerPanelOnMouseUp(Sender:TObject;
                                                     Button:TMouseButton;
                                                     Shift:TShiftState;
                                                     X,Y:Integer);
begin
  TrackPointerPanel.Catched:=False;
end;

procedure TCustomTrackBar.DoOnResize;
begin
  LeftArrowPanel.Left:=0;
  LeftArrowPanel.Top:=0;
  LeftArrowPanel.Height:=Height;

  RightArrowPanel.Left:=Width-RightArrowPanel.Width;
  RightArrowPanel.Top:=0;
  RightArrowPanel.Height:=Height;

  TrackPointerPanel.Top:=0;
  TrackPointerPanel.Height:=Height;

  SetValue(FValue);
end;

function TCustomTrackBar.GetCaption:TCaption;
begin
  Result:=TrackPointerPanel.Caption;
end;

procedure TCustomTrackBar.SetCaption(NewCaption:TCaption);
begin
  TrackPointerPanel.Caption:=NewCaption;

  inherited Caption:='';
end;

function TCustomTrackBar.GetFont:TFont;
begin
  LeftArrowPanel.Font:=TrackPointerPanel.Font;
  RightArrowPanel.Font:=TrackPointerPanel.Font;

  Result:=TrackPointerPanel.Font;
end;

procedure TCustomTrackBar.SetFont(NewFont:TFont);
begin
  LeftArrowPanel.Font:=NewFont;
  RightArrowPanel.Font:=NewFont;
  TrackPointerPanel.Font:=NewFont;
end;

function TCustomTrackBar.GetLeftArrowColor:TColor;
begin
  Result:=LeftArrowPanel.Color;
end;

procedure TCustomTrackBar.SetLeftArrowColor(NewColor:TColor);
begin
  LeftArrowPanel.Color:=NewColor;

  DoOnResize;
end;

function TCustomTrackBar.GetRightArrowColor:TColor;
begin
  Result:=RightArrowPanel.Color;
end;

procedure TCustomTrackBar.SetRightArrowColor(NewColor:TColor);
begin
  RightArrowPanel.Color:=NewColor;

  DoOnResize;
end;

function TCustomTrackBar.GetTrackPointerColor:TColor;
begin
  Result:=TrackPointerPanel.Color;
end;

procedure TCustomTrackBar.SetTrackPointerColor(NewColor:TColor);
begin
  TrackPointerPanel.Color:=NewColor;

  DoOnResize;
end;

function TCustomTrackBar.GetLeftArrowWidth:integer;
begin
  Result:=LeftArrowPanel.Width;
end;

procedure TCustomTrackBar.SetLeftArrowWidth(NewWidth:integer);
begin
  LeftArrowPanel.Width:=NewWidth;

  DoOnResize;
end;

function TCustomTrackBar.GetRightArrowWidth:integer;
begin
  Result:=RightArrowPanel.Width;
end;

procedure TCustomTrackBar.SetRightArrowWidth(NewWidth:integer);
begin
  RightArrowPanel.Width:=NewWidth;

  DoOnResize;
end;

function TCustomTrackBar.GetTrackPointerWidth:integer;
begin
  Result:=TrackPointerPanel.Width;
end;

procedure TCustomTrackBar.SetTrackPointerWidth(NewWidth:integer);
begin
  TrackPointerPanel.Width:=NewWidth;

  DoOnResize;
end;

procedure TCustomTrackBar.SetMinValue(NewValue:integer);
begin
  if NewValue >= MaxValue then
  FMinValue:=MaxValue-1
  else
  FMinValue:=NewValue;

  if NewValue < SAVE_MINIMUM then
  FMinValue:=SAVE_MINIMUM;

  if MinValue > Value then
  SetValue(MinValue)
  else
  SetValue(FValue);
end;

procedure TCustomTrackBar.SetMaxValue(NewValue:integer);
begin
  if NewValue <= MinValue then
  FMaxValue:=MinValue+1
  else
  FMaxValue:=NewValue;

  if NewValue > SAVE_MAXIMUM then
  FMaxValue:=SAVE_MAXIMUM;

  if MaxValue < Value then
  Value:=MaxValue
  else
  SetValue(FValue);
end;

procedure TCustomTrackBar.SetValue(NewValue:integer);
begin
  FValue:=NewValue;

  if FValue < MinValue then
  FValue:=MinValue;

  if FValue > MaxValue then
  FValue:=MaxValue;
  
  TrackPointerPanel.Left:=Round(
    ((Width-(LeftArrowPanel.Width+RightArrowPanel.Width+TrackPointerPanel.Width))/
     (MaxValue-MinValue))*
    (FValue-MinValue)+LeftArrowPanel.Width);

  if Assigned(OnChange) then OnChange(Self);
end;





procedure Register;
begin
  RegisterComponents('Game Components', [TCustomTrackBar]);
end;

end.
