unit PrivateListBox;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,StdCtrls;

type

  TUpArrowPanel = class(TPanel)
    private
      FOutlineColor:TColor;

      procedure SetOutlineColor(NewColor:TColor);
    protected
      procedure Paint;override;
    published
      property OutlineColor:TColor read FOutlineColor write SetOutlineColor;
  end;

  TDownArrowPanel = class(TPanel)
    private
      FOutlineColor:TColor;

      procedure SetOutlineColor(NewColor:TColor);
    protected
      procedure Paint;override;
    published
      property OutlineColor:TColor read FOutlineColor write SetOutlineColor;
  end;

  TPrivateListBox = class(TPanel)
    private
      FItems:TStrings;
      FItemIndex:integer;
      FItemHeight:integer;
      FArrowsWidth:integer;
      FArrowsHeight:integer;
      FColor:TColor;
      FOutlineColor:TColor;
      FSelItemTextColor:TColor;
      FSelItemBackgroundColor:TColor;

      LabelsPanel:TPanel;
      Labels:array of TLabel;
      UpArrowPanel:TUpArrowPanel;
      DownArrowPanel:TDownArrowPanel;

      Timer:TTimer;
      TimerIncValue:integer;
      MoveValue:integer;

      procedure SetItems(NewItems:TStrings);
      procedure SetItemIndex(NewIndex:integer);
      procedure SetItemHeight(NewHeight:integer);
      procedure SetArrowsWidth(NewWidth:integer);
      procedure SetArrowsHeight(NewHeight:integer);
      procedure SetColor(NewColor:TColor);
      procedure SetOutlineColor(NewColor:TColor);
      procedure SetSelItemTextColor(NewColor:TColor);
      procedure SetSelItemBackgroundColor(NewColor:TColor);


      procedure LabelOnMouseDown(Sender:TObject;
                                 Button:TMouseButton;
                                 Shift:TShiftState;
                                 X,Y:Integer);
      procedure UpArrowPanelOnMouseDown(Sender:TObject;
                                        Button:TMouseButton;
                                        Shift:TShiftState;
                                        X,Y:Integer);
      procedure DownArrowPanelOnMouseDown(Sender:TObject;
                                          Button:TMouseButton;
                                          Shift:TShiftState;
                                          X,Y:Integer);
      procedure UpArrowPanelOnDblClick(Sender:TObject);
      procedure DownArrowPanelOnDblClick(Sender:TObject);
      procedure TimerOnTimer(Sender:TObject);
      procedure StringsOnChange(Sender:TObject);
    protected
      procedure Paint;override;
      procedure Resize;override;
    public

      procedure UpdateControls;
      procedure MoveItemsDown;
      procedure MoveItemsUp;

      constructor Create(AOwner:TComponent);override;
      destructor Destroy;override;

    published
      property Items:TStrings read FItems write SetItems;
      property ItemIndex:integer read FItemIndex write SetItemIndex;
      property ItemHeight:integer read FItemHeight write SetItemHeight;
      property ArrowsWidth:integer read FArrowsWidth write SetArrowsWidth;
      property ArrowsHeight:integer read FArrowsHeight write SetArrowsHeight;
      property Color:TColor read FColor write SetColor;
      property OutlineColor:TColor read FOutlineColor write SetOutlineColor;
      property SelItemTextColor:TColor read FSelItemTextColor write SetSelItemTextColor;
      property SelItemBackgroundColor:TColor read FSelItemBackgroundColor write SetSelItemBackgroundColor;
  end;

procedure Register;

implementation

//*********  TUpArrowPanel  ****************************************************

procedure TUpArrowPanel.SetOutlineColor(NewColor:TColor);
begin
  FOutlineColor:=NewColor;

  Repaint;
end;

procedure TUpArrowPanel.Paint;
var Canvas:TCanvas;
    Points:array of TPoint;
begin
  inherited Paint;

  Canvas:=TCanvas.Create;
  Canvas.Handle:=GetDC(Handle);

  Canvas.Pen.Color:=OutlineColor;
  Canvas.Brush.Color:=Color;

  Canvas.Rectangle(0,0,Width,Height);

  SetLength(Points,3);
                 {
  Points[0].x:=Round(Width/2);
  Points[0].y:=Round(Height/3);
  Points[1].x:=Points[0].x-Round(Points[0].x/2);
  Points[1].y:=Points[0].y+Round(Points[0].x/2);
  Points[2].x:=Points[0].x+Round(Points[0].x/2);
  Points[2].y:=Points[0].y+Round(Points[0].x/2);     }

  Points[0].x:=Round(Width/2);
  Points[0].y:=Round(Height/3);
  Points[1].x:=Points[0].x-Round(Points[0].x/2);
  Points[1].y:=2*Round(Height/3);
  Points[2].x:=Points[0].x+Round(Points[0].x/2);
  Points[2].y:=2*Round(Height/3);

  Canvas.Polygon(Points);

  Canvas.Free;
end;



//*********  TDownArrowPanel  ****************************************************

procedure TDownArrowPanel.SetOutlineColor(NewColor:TColor);
begin
  FOutlineColor:=NewColor;

  Repaint;
end;

procedure TDownArrowPanel.Paint;
var Canvas:TCanvas;
    Points:array of TPoint;
begin
  inherited Paint;

  Canvas:=TCanvas.Create;
  Canvas.Handle:=GetDC(Handle);

  Canvas.Pen.Color:=OutlineColor;
  Canvas.Brush.Color:=Color;

  Canvas.Rectangle(0,0,Width,Height);

  SetLength(Points,3);
                                {
  Points[0].x:=Round(Width/2);
  Points[0].y:=2*Round(Height/3);
  Points[1].x:=Points[0].x-Round(Points[0].x/2);
  Points[1].y:=Points[0].y-Round(Points[0].x/2);
  Points[2].x:=Points[0].x+Round(Points[0].x/2);
  Points[2].y:=Points[0].y-Round(Points[0].x/2);      }

  Points[0].x:=Round(Width/2);
  Points[0].y:=2*Round(Height/3);
  Points[1].x:=Points[0].x-Round(Points[0].x/2);
  Points[1].y:=Round(Height/3);
  Points[2].x:=Points[0].x+Round(Points[0].x/2);
  Points[2].y:=Round(Height/3);

  Canvas.Polygon(Points);

  Canvas.Free;
end;







//*******  TPrivateListBox  ****************************************************

constructor TPrivateListBox.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);

  Width:=120;
  Height:=140;

  FItems:=TStringList.Create;
  TStringList(FItems).OnChange:=StringsOnChange;

  FItemIndex:=-1;

  FItemHeight:=13;

  LabelsPanel:=TPanel.Create(Self);
  LabelsPanel.Parent:=Self;
  LabelsPanel.BevelInner:=bvNone;
  LabelsPanel.BevelOuter:=bvNone;
  LabelsPanel.Left:=1;
  LabelsPanel.Top:=1;
  LabelsPanel.Caption:='';
  LabelsPanel.ParentColor:=True;
  LabelsPanel.ParentFont:=True;

  UpArrowPanel:=TUpArrowPanel.Create(Self);
  UpArrowPanel.Parent:=Self;
  UpArrowPanel.OnMouseDown:=UpArrowPanelOnMouseDown;
  UpArrowPanel.OnDblClick:=UpArrowPanelOnDblClick;
  DownArrowPanel:=TDownArrowPanel.Create(Self);
  DownArrowPanel.Parent:=Self;
  DownArrowPanel.OnMouseDown:=DownArrowPanelOnMouseDown;
  DownArrowPanel.OnDblClick:=DownArrowPanelOnDblClick;

  FArrowsWidth:=16;
  FArrowsHeight:=16;

  SetColor(clBlack);

  FSelItemTextColor:=clWhite;
  FSelItemBackgroundColor:=$00FFAF0F;

  Font.Color:=$00FFAF0F;

  SetOutlineColor($00FFAF0F);

  UpdateControls;

  Timer:=TTimer.Create(Self);
  Timer.Enabled:=False;
  Timer.Interval:=10;
  Timer.OnTimer:=TimerOnTimer;

end;

destructor TPrivateListBox.Destroy;
var C1:integer;
begin
  for C1:=0 to High(Labels) do
  Labels[C1].Free;

  SetLength(Labels,0);

  Timer.Free;

  inherited Destroy;
end;


procedure TPrivateListBox.SetItems(NewItems:TStrings);
begin
  FItems.Assign(NewItems);
end;

procedure TPrivateListBox.SetItemIndex(NewIndex:integer);
var C1:integer;
begin
  if NewIndex > Length(Labels)-1 then Exit;

  FItemIndex:=NewIndex;

  if (Low(Labels) <= NewIndex) and (High(Labels) >= NewIndex) then
  for C1:=0 to High(Labels) do
  if C1 = NewIndex then
  begin
    Labels[C1].Font.Color:=SelItemTextColor;
    Labels[C1].Color:=SelItemBackgroundColor;
  end
  else
  begin
    Labels[C1].ParentFont:=True;
    Labels[C1].ParentColor:=True;
  end;
end;

procedure TPrivateListBox.SetItemHeight(NewHeight:integer);
begin
  FItemHeight:=NewHeight;

  UpdateControls;
end;

procedure TPrivateListBox.SetArrowsWidth(NewWidth:integer);
begin
  FArrowsWidth:=NewWidth;

  UpdateControls;
end;

procedure TPrivateListBox.SetArrowsHeight(NewHeight:integer);
begin
  FArrowsHeight:=NewHeight;

  UpdateControls;
end;

procedure TPrivateListBox.SetColor(NewColor:TColor);
begin
  FColor:=NewColor;

  inherited Color:=NewColor;

  LabelsPanel.Color:=NewColor;

  UpArrowPanel.Color:=NewColor;
  DownArrowPanel.Color:=NewColor;
end;

procedure TPrivateListBox.SetOutlineColor(NewColor:TColor);
begin
  FOutlineColor:=NewColor;

  UpArrowPanel.OutlineColor:=NewColor;
  DownArrowPanel.OutlineColor:=NewColor;

  Repaint;
end;

procedure TPrivateListBox.SetSelItemTextColor(NewColor:TColor);
begin
  FSelItemTextColor:=NewColor;

  SetItemIndex(ItemIndex);
end;

procedure TPrivateListBox.SetSelItemBackgroundColor(NewColor:TColor);
begin
  FSelItemBackgroundColor:=NewColor;

  SetItemIndex(ItemIndex);
end;

procedure TPrivateListBox.UpdateControls;
var C1:integer;
begin

  LabelsPanel.Width:=Width-ArrowsWidth-1;
  LabelsPanel.Height:=Height-2;

  for C1:=0 to High(Labels) do
  Labels[C1].Free;

  SetLength(Labels,Items.Count);

  for C1:=0 to Items.Count-1 do
  begin
    Labels[C1]:=TLabel.Create(LabelsPanel);
    Labels[C1].Parent:=LabelsPanel;
    Labels[C1].ParentFont:=True;
    Labels[C1].Left:=1;

    if C1 = 0 then
    Labels[C1].Top:=1
    else
    Labels[C1].Top:=Labels[C1-1].Top+Labels[C1-1].Height;

    Labels[C1].Width:=Width-ArrowsWidth;
    Labels[C1].Height:=ItemHeight;
    Labels[C1].AutoSize:=False;
    Labels[C1].Caption:=Items.Strings[C1];
    Labels[C1].OnMouseDown:=LabelOnMouseDown;
    Labels[C1].Tag:=C1;
  end;

  UpArrowPanel.Width:=ArrowsWidth;
  UpArrowPanel.Height:=ArrowsHeight;
  UpArrowPanel.Left:=Width-ArrowsWidth;
  UpArrowPanel.Top:=0;

  DownArrowPanel.Width:=ArrowsWidth;
  DownArrowPanel.Height:=ArrowsHeight;
  DownArrowPanel.Left:=Width-ArrowsWidth;
  DownArrowPanel.Top:=Height-ArrowsHeight;

end;

procedure TPrivateListBox.MoveItemsDown;
begin
  if Length(Labels) <= 0 then Exit;

  if Labels[0].Top > 0 then Exit;

  MoveValue:=0;

  TimerIncValue:=1;

  Timer.Enabled:=True;

  repeat
    Application.ProcessMessages;
  until MoveValue >= ItemHeight;

  Timer.Enabled:=False;
end;

procedure TPrivateListBox.MoveItemsUp;
begin
  if Length(Labels) <= 0 then Exit;
                          
  if Labels[High(Labels)].Top+Labels[High(Labels)].Height < Height then Exit;

  MoveValue:=0;

  TimerIncValue:=-1;

  Timer.Enabled:=True;

  repeat
    Application.ProcessMessages;
  until MoveValue >= ItemHeight;

  Timer.Enabled:=False;
end;

procedure TPrivateListBox.LabelOnMouseDown(Sender:TObject;
                                           Button:TMouseButton;
                                           Shift:TShiftState;
                                           X,Y:Integer);
begin
  SetItemIndex(TLabel(Sender).Tag);
end;

procedure TPrivateListBox.UpArrowPanelOnMouseDown(Sender:TObject;
                                                  Button:TMouseButton;
                                                  Shift:TShiftState;
                                                  X,Y:Integer);
begin
  MoveItemsDown;
end;

procedure TPrivateListBox.DownArrowPanelOnMouseDown(Sender:TObject;
                                                    Button:TMouseButton;
                                                    Shift:TShiftState;
                                                    X,Y:Integer);
begin
  MoveItemsUp;
end;

procedure TPrivateListBox.UpArrowPanelOnDblClick(Sender:TObject);
var C1:integer;
begin
  for C1:=0 to Round(Items.Count/10) do
  MoveItemsDown;
end;

procedure TPrivateListBox.DownArrowPanelOnDblClick(Sender:TObject);
var C1:integer;
begin
  for C1:=0 to Round(Items.Count/10) do
  MoveItemsUp;
end;

procedure TPrivateListBox.TimerOnTimer(Sender:TObject);
var C1:integer;
begin
  Inc(MoveValue);

  for C1:=0 to High(Labels) do
  Labels[C1].Top:=Labels[C1].Top+TimerIncValue;
end;

procedure TPrivateListBox.StringsOnChange(Sender:TObject);
begin
  UpdateControls;
end;

procedure TPrivateListBox.Paint;
var Canvas:TCanvas;
begin
  inherited Paint;

  Canvas:=TControlCanvas.Create;
  Canvas.Handle:=GetDC(Handle);

  Canvas.Pen.Color:=OutlineColor;
  Canvas.Brush.Color:=Color;

  Canvas.Rectangle(0,0,Width,Height);

  Canvas.MoveTo(Width-ArrowsWidth,0);
  Canvas.LineTo(Width-ArrowsWidth,Height);

  Canvas.Free;

end;

procedure TPrivateListBox.Resize;
begin
  inherited Resize;

  UpdateControls;
end;




procedure Register;
begin
  RegisterComponents('Game Components', [TPrivateListBox]);
end;

end.
