unit PrivateRadioGroup;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  ExtCtrls,StdCtrls;

type
  TPrivateRadioGroup = class(TPanel)
    private
      FCaption:TCaption;
      FCaptionHeight:integer;
      FCaptionUnderline:boolean;
      FItems:TStrings;
      FItemIndex:integer;
      FSelItemColor:TColor;
      FItemsLeft:integer;
      FOutlineColor:TColor;

      FOnChange:TNotifyEvent;

      Labels:array of TLabel;

      procedure SetItemIndex(NewItemIndex:integer);
      procedure SetItems(NewItems:TStrings);
      procedure SetCaption(NewCaption:TCaption);
      procedure SetCaptionHeight(NewHeight:integer);
      procedure SetCaptionUnderline(NewValue:boolean);
      procedure SetSelItemColor(NewColor:TColor);
      procedure SetItemsLeft(NewValue:integer);
      procedure SetOutlineColor(NewColor:TColor);

      procedure StringsOnChange(Sender:TObject);
      procedure LabelOnMouseDown(Sender:TObject;
                                 Button:TMouseButton;
                                 Shift:TShiftState;
                                 X,Y:Integer);
    protected
      procedure Paint;override;

    public

      constructor Create(AOwner:TComponent);override;
      destructor Destroy;override;

      procedure UpdateButtons;

    published
      property Caption:TCaption read FCaption write SetCaption;
      property CaptionHeight:integer read FCaptionHeight write SetCaptionHeight;
      property CaptionUnderline:boolean read FCaptionUnderline write SetCaptionUnderline;
      property Items:TStrings read FItems write SetItems;
      property ItemIndex:integer read FItemIndex write SetItemIndex;
      property SelItemColor:TColor read FSelItemColor write SetSelItemColor;
      property ItemsLeft:integer read FItemsLeft write SetItemsLeft;
      property OutlineColor:TColor read FOutlineColor write SetOutlineColor;

      property OnChange:TNotifyEvent read FOnChange write FOnChange;


  end;

procedure Register;

implementation

constructor TPrivateRadioGroup.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);

  FItems:=TStringList.Create;
  TStringList(FItems).OnChange:=StringsOnChange;

  FCaptionHeight:=20;

  FItemsLeft:=10;

  FSelItemColor:=clWhite;

  FItemIndex:=-1;
end;

destructor TPrivateRadioGroup.Destroy;
var C1:integer;
begin
  for C1:=0 to Items.Count-1 do
  Labels[C1].Free;

  FItems.Free;

  inherited Destroy;
end;

procedure TPrivateRadioGroup.UpdateButtons;
var C1:integer;
begin
  for C1:=0 to High(Labels) do
  Labels[C1].Free;

  SetLength(Labels,Items.Count);

  for C1:=0 to Items.Count-1 do
  begin
    Labels[C1]:=TLabel.Create(Self);
    Labels[C1].Parent:=Self;
    Labels[C1].Left:=ItemsLeft;

    if C1 = 0 then
    Labels[C1].Top:=CaptionHeight+Round(((Height-CaptionHeight)-Items.Count*Labels[0].Height)/(Items.Count+1))
    else
    Labels[C1].Top:=Labels[C1-1].Top+Labels[C1-1].Height+Round(((Height-CaptionHeight)-Items.Count*Labels[0].Height)/(Items.Count+1));

    Labels[C1].Transparent:=True;
    Labels[C1].Caption:=Items.Strings[C1];
    Labels[C1].OnMouseDown:=LabelOnMouseDown;
    Labels[C1].Tag:=C1;
    Labels[C1].Visible:=True;
  end;

  SetItemIndex(-1);                    

end;

procedure TPrivateRadioGroup.SetItemIndex(NewItemIndex:integer);
var C1:integer;
begin
  FItemIndex:=NewItemIndex;

  for C1:=0 to Items.Count-1 do
  begin
    if C1 <> NewItemIndex then
    Labels[C1].ParentFont:=True
    else
    Labels[C1].Font.Color:=SelItemColor;
  end;
end;

procedure TPrivateRadioGroup.SetItems(NewItems:TStrings);
begin
  FItems.Assign(NewItems);

  UpdateButtons;
end;

procedure TPrivateRadioGroup.SetCaption(NewCaption:TCaption);
begin
  inherited Caption:='';

  FCaption:=NewCaption;

  Repaint;
end;

procedure TPrivateRadioGroup.SetCaptionHeight(NewHeight:integer);
begin
  FCaptionHeight:=NewHeight;

  Repaint;
end;

procedure TPrivateRadioGroup.SetCaptionUnderline(NewValue:boolean);
begin
  FCaptionUnderline:=NewValue;

  Repaint;
end;

procedure TPrivateRadioGroup.SetSelItemColor(NewColor:TColor);
begin
  FSelItemColor:=NewColor;

  if ItemIndex >= 0 then
  Labels[ItemIndex].Font.Color:=NewColor;
end;

procedure TPrivateRadioGroup.SetItemsLeft(NewValue:integer);
begin
  FItemsLeft:=NewValue;

  UpdateButtons;
end;

procedure TPrivateRadioGroup.SetOutlineColor(NewColor:TColor);
begin
  FOutlineColor:=NewColor;

  Repaint;
end;

procedure TPrivateRadioGroup.StringsOnChange(Sender:TObject);
begin
  UpdateButtons;
end;

procedure TPrivateRadioGroup.LabelOnMouseDown(Sender:TObject;
                                              Button:TMouseButton;
                                              Shift:TShiftState;
                                              X,Y:Integer);
begin
  SetItemIndex(TLabel(Sender).Tag);

  if Assigned(OnChange) then OnChange(Self);
end;

procedure TPrivateRadioGroup.Paint;
var Canvas:TCanvas;
    C1:integer;
begin
  inherited Paint;

  Canvas:=TControlCanvas.Create;
  Canvas.Handle:=GetDC(Handle);

  Canvas.Pen.Color:=OutlineColor;
  Canvas.Brush.Color:=Color;

  Canvas.Rectangle(0,0,Width,Height);

  if CaptionUnderline then
  begin
    Canvas.MoveTo(0,CaptionHeight);
    Canvas.LineTo(Width,CaptionHeight);
  end;

  Canvas.Font:=Font;

  with Canvas do
  TextOut(5,Round((CaptionHeight-TextExtent(Caption).cy)/2),Caption);

  for C1:=0 to High(Labels) do
  begin
    if C1 = ItemIndex then
    Canvas.Font.Color:=SelItemColor
    else
    Canvas.Font.Color:=Font.Color;

    Canvas.TextOut(Labels[C1].Left,Labels[C1].Top,Labels[C1].Caption);
  end;

  Canvas.Free;

end;






procedure Register;
begin
  RegisterComponents('Game Components', [TPrivateRadioGroup]);
end;

end.