unit RainbowBtn;

interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls,ExtCtrls;

type

  TRainbowBtn = class(TPanel)
    private
      FMouseOn:boolean;

      FCaption:TCaption;

      FNormalTopColor:TColor;
      FNormalBottomColor:TColor;
      FMouseOnTopColor:TColor;
      FMouseOnBottomColor:TColor;
      FOnMouseEnter:TNotifyEvent;
      FOnMouseLeave:TNotifyEvent;
      FOnResize:TNotifyEvent;

      NormalBitmap:TBitmap;
      MouseOnBitmap:TBitmap;
      TargetBitmap:TBitmap;
                            
      procedure SetCaption(NewCaption:TCaption);

      procedure SetNormalTopColor(NewColor:TColor);
      procedure SetNormalBottomColor(NewColor:TColor);
      procedure SetMouseOnTopColor(NewColor:TColor);
      procedure SetMouseOnBottomColor(NewColor:TColor);

      procedure PanelOnResize(Sender:TObject);

    protected
      procedure Paint;override;

      procedure MouseEnter(var Message: TMessage); message CM_MOUSEENTER;
      procedure MouseLeave(var Message: TMessage); message CM_MOUSELEAVE;
    public
      property MouseOn:boolean read FMouseOn;

      constructor Create(AOwner:TComponent);override;
      destructor Destroy;override;

      procedure DoOnResize;
      procedure RepaintNormalBitmap;
      procedure RepaintMouseOnBitmap;
    published
      property Caption:TCaption read FCaption write SetCaption;
      property NormalTopColor:TColor read FNormalTopColor write SetNormalTopColor;
      property NormalBottomColor:TColor read FNormalBottomColor write SetNormalBottomColor;
      property MouseOnTopColor:TColor read FMouseOnTopColor write SetMouseOnTopColor;
      property MouseOnBottomColor:TColor read FMouseOnBottomColor write SetMouseOnBottomColor;
      property OnMouseEnter:TNotifyEvent read FOnMouseEnter write FOnMouseEnter;
      property OnMouseLeave:TNotifyEvent read FOnMouseLeave write FOnMouseLeave;
      property OnResize:TNotifyEvent read FOnResize write FOnResize;

  end;

procedure Register;

implementation

//*******  TRainbowBtn  *****************************************************

constructor TRainbowBtn.Create(AOwner:TComponent);
begin
  inherited Create(AOwner);
  Parent:=TWinControl(AOwner);
  BevelInner:=bvLowered;
  BevelOuter:=bvRaised;
  BevelWidth:=1;

  NormalBitmap:=TBitmap.Create;
  MouseOnBitmap:=TBitmap.Create;
  TargetBitmap:=TBitmap.Create;

  Font.Name:='MS Sans Serif';
  Font.Size:=10;
  Font.Style:=Font.Style-[fsBold,fsItalic];
  Font.Color:=clWhite;
                        
  FNormalTopColor:=$00FFCF8F;
  FNormalBottomColor:=$00FF6F3F;
  FMouseOnTopColor:=$00FFEFAF;
  FMouseOnBottomColor:=$00FF9F3F;

  DoOnResize;

  inherited OnResize:=PanelOnResize;

end;

destructor TRainbowBtn.Destroy;
begin
  NormalBitmap.Free;
  MouseOnBitmap.Free;
  TargetBitmap.Free;

  inherited Destroy;
end;

procedure TRainbowBtn.Paint;
var HandleDC:HDC;
begin
  inherited Paint;

  HandleDC:=GetDC(Handle);

  TargetBitmap.Canvas.Font:=Font;

  with TargetBitmap.Canvas do
  begin

    if not MouseOn then
    CopyRect(Rect(0,0,Width,Height),
             NormalBitmap.Canvas,
             Rect(0,0,Width,Height))
    else
    CopyRect(Rect(0,0,Width,Height),
             MouseOnBitmap.Canvas,
             Rect(0,0,Width,Height));

    Brush.Style:=bsClear;

    TextOut(Round((Width-TextExtent(Caption).cx)/2),
            Round((Height-TextExtent(Caption).cy)/2),
            Caption);

    BitBlt(HandleDC,
           2,
           2,
           Width-4,
           Height-4,
           Handle,
           2,
           2,
           SRCCOPY);
  end;

end;

procedure TRainbowBtn.PanelOnResize(Sender:TObject);
begin
  DoOnResize;

  if Assigned(OnResize) then OnResize(Self);
end;

procedure TRainbowBtn.MouseEnter(var Message: TMessage);
begin
  FMouseOn:=True;

  if Assigned(OnMouseEnter) then OnMouseEnter(Self);

  Repaint;
end;

procedure TRainbowBtn.MouseLeave(var Message: TMessage);
begin
  FMouseOn:=False;

  if Assigned(OnMouseLeave) then OnMouseLeave(Self);

  Repaint;
end;

procedure TRainbowBtn.DoOnResize;
begin
  TargetBitmap.Width:=Width;
  TargetBitmap.Height:=Height;
  TargetBitmap.PixelFormat:=pf24bit;

  RepaintNormalBitmap;
  RepaintMouseOnBitmap;
  Repaint;
end;

procedure TRainbowBtn.RepaintNormalBitmap;
type TColorRGBA = record
       R:byte;
       G:byte;
       B:byte;
       A:byte;
     end;
var TopColorRGBA,BottomColorRGBA:TColorRGBA;
    TopColorDW:DWORD absolute TopColorRGBA;
    BottomColorDW:DWORD absolute BottomColorRGBA;
    DestColorRGBA:TColorRGBA;
    DestColorDW:DWORD absolute DestColorRGBA;
    Ratio:double;
    C1,C2:integer;
begin

  NormalBitmap.Width:=Width;
  NormalBitmap.Height:=Height;
  NormalBitmap.PixelFormat:=pf24bit;

  TopColorDW:=NormalTopColor;
  BottomColorDW:=NormalBottomColor;

  DestColorRGBA.A:=0;

  for C1:=0 to NormalBitmap.Height-1 do
  begin
    Ratio:=Abs(TopColorRGBA.R-BottomColorRGBA.R)/NormalBitmap.Height;
    if (TopColorRGBA.R > BottomColorRGBA.R) then
    DestColorRGBA.R:=Round(TopColorRGBA.R-C1*Ratio)
    else
    DestColorRGBA.R:=Round(TopColorRGBA.R+C1*Ratio);

    Ratio:=Abs(TopColorRGBA.G-BottomColorRGBA.G)/NormalBitmap.Height;
    if (TopColorRGBA.G > BottomColorRGBA.G) then
    DestColorRGBA.G:=Round(TopColorRGBA.G-C1*Ratio)
    else
    DestColorRGBA.G:=Round(TopColorRGBA.G+C1*Ratio);

    Ratio:=Abs(TopColorRGBA.B-BottomColorRGBA.B)/NormalBitmap.Height;
    if (TopColorRGBA.B > BottomColorRGBA.B) then
    DestColorRGBA.B:=Round(TopColorRGBA.B-C1*Ratio)
    else
    DestColorRGBA.B:=Round(TopColorRGBA.B+C1*Ratio);

    for C2:=0 to NormalBitmap.Width-1 do
    begin
      NormalBitmap.Canvas.Pixels[C2,C1]:=DestColorDW;
    end;
  end;

end;

procedure TRainbowBtn.RepaintMouseOnBitmap;
type TColorRGBA = record
       R:byte;
       G:byte;
       B:byte;
       A:byte;
     end;
var TopColorRGBA,BottomColorRGBA:TColorRGBA;
    TopColorDW:DWORD absolute TopColorRGBA;
    BottomColorDW:DWORD absolute BottomColorRGBA;
    DestColorRGBA:TColorRGBA;
    DestColorDW:DWORD absolute DestColorRGBA;
    Ratio:double;
    C1,C2:integer;
begin

  MouseOnBitmap.Width:=Width;
  MouseOnBitmap.Height:=Height;
  MouseOnBitmap.PixelFormat:=pf24bit;


  TopColorDW:=MouseOnTopColor;
  BottomColorDW:=MouseOnBottomColor;

  DestColorRGBA.A:=0;

  for C1:=0 to MouseOnBitmap.Height-1 do
  begin
    Ratio:=Abs(TopColorRGBA.R-BottomColorRGBA.R)/MouseOnBitmap.Height;
    if (TopColorRGBA.R > BottomColorRGBA.R) then
    DestColorRGBA.R:=Round(TopColorRGBA.R-C1*Ratio)
    else
    DestColorRGBA.R:=Round(TopColorRGBA.R+C1*Ratio);

    Ratio:=Abs(TopColorRGBA.G-BottomColorRGBA.G)/MouseOnBitmap.Height;
    if (TopColorRGBA.G > BottomColorRGBA.G) then
    DestColorRGBA.G:=Round(TopColorRGBA.G-C1*Ratio)
    else
    DestColorRGBA.G:=Round(TopColorRGBA.G+C1*Ratio);

    Ratio:=Abs(TopColorRGBA.B-BottomColorRGBA.B)/MouseOnBitmap.Height;
    if (TopColorRGBA.B > BottomColorRGBA.B) then
    DestColorRGBA.B:=Round(TopColorRGBA.B-C1*Ratio)
    else
    DestColorRGBA.B:=Round(TopColorRGBA.B+C1*Ratio);

    for C2:=0 to MouseOnBitmap.Width-1 do
    begin
      MouseOnBitmap.Canvas.Pixels[C2,C1]:=DestColorDW;
    end;
  end;

end;

procedure TRainbowBtn.SetCaption(NewCaption:TCaption);
begin
  FCaption:=NewCaption;

  inherited Caption:=NewCaption;
end;

procedure TRainbowBtn.SetNormalTopColor(NewColor:TColor);
begin
  FNormalTopColor:=NewColor;

  RepaintNormalBitmap;

  Repaint;
end;

procedure TRainbowBtn.SetNormalBottomColor(NewColor:TColor);
begin
  FNormalBottomColor:=NewColor;

  RepaintNormalBitmap;

  Repaint;
end;

procedure TRainbowBtn.SetMouseOnTopColor(NewColor:TColor);
begin
  FMouseOnTopColor:=NewColor;

  RepaintMouseOnBitmap;

  Repaint;
end;

procedure TRainbowBtn.SetMouseOnBottomColor(NewColor:TColor);
begin
  FMouseOnBottomColor:=NewColor;

  RepaintMouseOnBitmap;

  Repaint;
end;                   


procedure Register;
begin
  RegisterComponents('Game Components', [TRainbowBtn]);
end;

end.
